<?php
/*
Plugin Name: Alt Text AI
Description: AI Alt Text Generator
Version: 0.9
Author: Smartboost
*/

if (!defined('ABSPATH')) exit;

/* ---------------- Option keys ---------------- */
const CTAI_LITE_OPT_KEY         = 'ctai_lite_api_key';
const CTAI_LITE_OPT_CHARS       = 'ctai_lite_max_chars';
const CTAI_LITE_OPT_PROMPT      = 'ctai_lite_prompt';
const CTAI_LITE_OPT_USE_META    = 'ctai_lite_use_metadata';
const CTAI_LITE_OPT_META_TYPE   = 'ctai_lite_metadata_type'; // 'post_title' | 'category'
const CTAI_LITE_OPT_MAX_WIDTH   = 'ctai_lite_max_width';     // used to pick a rendition URL
const CTAI_LITE_OPT_MONTHLY     = 'ctai_lite_monthly_totals';// array('YYYY-MM' => float)
const CTAI_LITE_OPT_SKIP_IF_ALT = 'ctai_lite_skip_if_has_alt';

/* ---------------- Attachment meta keys (logging) ---------------- */
const CTAI_LITE_AMK_ALT       = '_wp_attachment_image_alt';
const CTAI_LITE_AMK_TOK_IN    = '_ctai_lite_prompt_tokens';
const CTAI_LITE_AMK_TOK_OUT   = '_ctai_lite_completion_tokens';
const CTAI_LITE_AMK_COST      = '_ctai_lite_cost_usd';
const CTAI_LITE_AMK_STATUS    = '_ctai_lite_status';         // success|error|skipped
const CTAI_LITE_AMK_LAST_ERR  = '_ctai_lite_last_error';
const CTAI_LITE_AMK_LAST_TS   = '_ctai_lite_last_ts';
const CTAI_LITE_AMK_SENT_URL  = '_ctai_lite_sent_url';
const CTAI_LITE_AMK_SENT_W    = '_ctai_lite_sent_width';

/* ---------------- Pricing (USD per token) ---------------- */
add_filter('ctai_lite_rate_in',  fn() => 0.00000015); // $0.15 / 1M input
add_filter('ctai_lite_rate_out', fn() => 0.00000060); // $0.60 / 1M output

/* ---------------- Bulk generation (queue) ---------------- */
const CTAI_LITE_OPT_BULK = 'ctai_lite_bulk_state'; // stores queue + status

function ctai_lite_bulk_get_state(): array {
  $st = get_option(CTAI_LITE_OPT_BULK, []);
  $st += [
    'status'     => 'idle', // idle|running|paused|cancelling|done|error
    'queue'      => [],     // array of attachment IDs
    'total'      => 0,
    'processed'  => 0,
    'skipped'    => 0,
    'errors'     => 0,
    'current_id' => null,
    'started_at' => 0,
    'last_ts'    => 0,
    'log'        => [],
  ];
  if (count($st['log']) > 20) $st['log'] = array_slice($st['log'], -20);
  return $st;
}
function ctai_lite_bulk_set_state(array $st): void {
  if (count($st['log']) > 20) $st['log'] = array_slice($st['log'], -20);
  update_option(CTAI_LITE_OPT_BULK, $st, false);
}
function ctai_lite_bulk_log(string $msg): void {
  $st = ctai_lite_bulk_get_state();
  $st['log'][] = wp_date('H:i:s') . ' — ' . $msg;
  ctai_lite_bulk_set_state($st);
}

/* ---------------- Defaults ---------------- */
function ctai_lite_defaults() {
  return [
    CTAI_LITE_OPT_CHARS       => 100,
    CTAI_LITE_OPT_PROMPT      => 'Generate concise, literal alt text for screen readers (max {MAX_CHARS} characters). Neutral tone. No brand names, no marketing claims.',
    CTAI_LITE_OPT_USE_META    => 0,
    CTAI_LITE_OPT_META_TYPE   => 'post_title',
    CTAI_LITE_OPT_MAX_WIDTH   => 500, // default width
    CTAI_LITE_OPT_SKIP_IF_ALT => 1,
  ];
}

/* ---------------- Timezone helpers ---------------- */
function ctai_lite_tz() : DateTimeZone {
  if (function_exists('wp_timezone_string')) {
    $tz = wp_timezone_string();
    if ($tz && @timezone_open($tz)) return new DateTimeZone($tz);
  }
  return new DateTimeZone('America/Los_Angeles');
}
function ctai_lite_current_month_range(): array {
  $tz = ctai_lite_tz();
  $now = new DateTime('now', $tz);
  $start = new DateTime($now->format('Y-m-01') . ' 00:00:00', $tz);
  $end = clone $now;
  return [$start->getTimestamp(), $end->getTimestamp(), $start->format('Y-m-d'), $end->format('Y-m-d')];
}
function ctai_lite_parse_date_range_from_request(): array {
  [$def_start_ts, $def_end_ts, $def_start_str, $def_end_str] = ctai_lite_current_month_range();
  $tz = ctai_lite_tz();
  $from_s = isset($_GET['ctai_from']) ? sanitize_text_field($_GET['ctai_from']) : $def_start_str;
  $to_s   = isset($_GET['ctai_to'])   ? sanitize_text_field($_GET['ctai_to'])   : $def_end_str;
  $from_dt = DateTime::createFromFormat('Y-m-d H:i:s', $from_s . ' 00:00:00', $tz);
  $to_dt   = DateTime::createFromFormat('Y-m-d H:i:s', $to_s   . ' 23:59:59',  $tz);
  $from_ts = $from_dt ? $from_dt->getTimestamp() : $def_start_ts;
  $to_ts   = $to_dt   ? $to_dt->getTimestamp()   : $def_end_ts;
  if ($from_ts > $to_ts) { $t = $from_ts; $from_ts = $to_ts; $to_ts = $t; }
  return [$from_ts, $to_ts, $from_s, $to_s];
}

/* ---------------- Settings page ---------------- */
add_action('admin_menu', function () {
  add_options_page('Alt Text AI', 'Alt Text AI', 'manage_options', 'ctai-lite', 'ctai_lite_settings_page');
});

function ctai_lite_settings_page() {
  if (!current_user_can('manage_options')) return;
  $d = ctai_lite_defaults();

  /* save */
  if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'ctai_lite_save')) {

    // API key: only overwrite if a non-empty value was provided
    $posted_key = trim((string)($_POST['ctai_lite_api_key'] ?? ''));
    if ($posted_key !== '') {
        update_option(CTAI_LITE_OPT_KEY, sanitize_text_field($posted_key));
    }

    update_option(CTAI_LITE_OPT_PROMPT,     sanitize_text_field($_POST['ctai_lite_prompt'] ?? $d[CTAI_LITE_OPT_PROMPT]));
    update_option(CTAI_LITE_OPT_USE_META,   isset($_POST['ctai_lite_use_meta']) ? 1 : 0);
    $mt = in_array($_POST['ctai_lite_meta_type'] ?? 'post_title', ['post_title','category'], true) ? $_POST['ctai_lite_meta_type'] : 'post_title';
    update_option(CTAI_LITE_OPT_META_TYPE,  $mt);
    update_option(CTAI_LITE_OPT_MAX_WIDTH,  max(320, (int)($_POST['ctai_lite_max_width'] ?? $d[CTAI_LITE_OPT_MAX_WIDTH])));
    update_option(CTAI_LITE_OPT_CHARS,      max(40,  (int)($_POST['ctai_lite_max_chars'] ?? $d[CTAI_LITE_OPT_CHARS])));
    update_option(CTAI_LITE_OPT_SKIP_IF_ALT, isset($_POST['ctai_lite_skip']) ? 1 : 0);

    echo '<div class="updated"><p>Saved.</p></div>';
  }

  // read settings for display
  $has_key = trim((string) get_option(CTAI_LITE_OPT_KEY, '')) !== '';
  $prmpt = get_option(CTAI_LITE_OPT_PROMPT,  $d[CTAI_LITE_OPT_PROMPT]);
  $usemd = (int) get_option(CTAI_LITE_OPT_USE_META, $d[CTAI_LITE_OPT_USE_META]);
  $mtype = get_option(CTAI_LITE_OPT_META_TYPE, $d[CTAI_LITE_OPT_META_TYPE]);
  $width = (int) get_option(CTAI_LITE_OPT_MAX_WIDTH, $d[CTAI_LITE_OPT_MAX_WIDTH]);
  $chars = (int) get_option(CTAI_LITE_OPT_CHARS, $d[CTAI_LITE_OPT_CHARS]);
  $skip  = (int) get_option(CTAI_LITE_OPT_SKIP_IF_ALT, $d[CTAI_LITE_OPT_SKIP_IF_ALT]);

  // connection-test notices
  if (isset($_GET['ctai_lite_msg'])) {
    $msg = sanitize_key($_GET['ctai_lite_msg']);
    $detail = isset($_GET['detail']) ? esc_html(rawurldecode($_GET['detail'])) : '';
    if     ($msg==='ok')       echo '<div class="notice notice-success is-dismissible"><p>Connection test OK.</p></div>';
    elseif ($msg==='no_key')   echo '<div class="notice notice-warning is-dismissible"><p>Add your API key, then retry the test.</p></div>';
    elseif ($msg==='err_http') echo '<div class="notice notice-error is-dismissible"><p>HTTP error: '.$detail.'</p></div>';
    elseif ($msg==='err_api')  echo '<div class="notice notice-error is-dismissible"><p>OpenAI responded but not as expected. Check the key/permissions.</p></div>';
  }

  // monthly total (keyed by YYYY-MM, naturally resets on 1st)
  $totals = (array) get_option(CTAI_LITE_OPT_MONTHLY, []);
  $ym     = wp_date('Y-m', null, ctai_lite_tz());
  $month_total = isset($totals[$ym]) ? (float)$totals[$ym] : 0.0;

  // date-range filter for table
  [$from_ts, $to_ts, $from_str, $to_str] = ctai_lite_parse_date_range_from_request();

  ?>
<div class="wrap">
    <h1>Alt Text AI</h1>

    <form method="post">
        <?php wp_nonce_field('ctai_lite_save'); ?>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><label>OpenAI API key</label></th>
                <td>
                    <input type="password" name="ctai_lite_api_key" value="" placeholder="<?php echo $has_key ? 'Saved — leave blank to keep' : ''; ?>" autocomplete="new-password" style="width:560px" />
                    <p class="description">
                        <?php echo $has_key
                      ? 'A key is already saved. Leave this blank to keep it, or enter a new key to replace.'
                      : 'Paste your OpenAI key. It will be stored in the database.'; ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><label>Prompt</label></th>
                <td>
                    <input type="text" name="ctai_lite_prompt" value="<?php echo esc_attr($prmpt); ?>" size="80">
                    <p class="description">Use <code>{MAX_CHARS}</code> to insert the limit below.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Append metadata</th>
                <td>
                    <label><input type="checkbox" name="ctai_lite_use_meta" value="1" <?php checked($usemd,1); ?>> Enable</label>
                    <p>
                        <label>Type:
                            <select name="ctai_lite_meta_type">
                                <option value="post_title" <?php selected($mtype,'post_title'); ?>>Parent Post Title</option>
                                <option value="category" <?php selected($mtype,'category');  ?>>First Category</option>
                            </select>
                        </label>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><label>Skip if alt already exists</label></th>
                <td>
                    <label><input type="checkbox" name="ctai_lite_skip" value="1" <?php checked($skip,1); ?>> Don’t overwrite existing alt text</label>
                </td>
            </tr>
            <tr>
                <th scope="row"><label>Resize width sent to API</label></th>
                <td>
                    <input type="number" name="ctai_lite_max_width" value="<?php echo esc_attr($width); ?>" min="320" step="10"> px
                    <p class="description">Sends an existing WordPress rendition ≤ this width if available (e.g., 768px). If none exists, falls back to the closest size, then original.</p>
                </td>
            </tr>
            <tr>
                <th scope="row"><label>Max characters</label></th>
                <td><input type="number" name="ctai_lite_max_chars" value="<?php echo esc_attr($chars); ?>" min="40" step="5"></td>
            </tr>
        </table>
        <?php submit_button('Save Settings'); ?>
    </form>

    <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
        <?php wp_nonce_field('ctai_lite_test'); ?>
        <input type="hidden" name="action" value="ctai_lite_test_connection">
        <?php submit_button('Run Connection Test', 'secondary'); ?>
    </form>

    <h2>Usage This Month</h2>
    <p><strong><?php echo esc_html($ym); ?></strong> total cost: <strong>$<?php echo number_format($month_total, 4); ?></strong></p>

    <form method="get" style="margin:12px 0;">
        <input type="hidden" name="page" value="ctai-lite">
        <label style="margin-right:8px;">From:
            <input type="date" name="ctai_from" value="<?php echo esc_attr($from_str); ?>">
        </label>
        <label style="margin-right:8px;">To:
            <input type="date" name="ctai_to" value="<?php echo esc_attr($to_str); ?>">
        </label>
        <?php submit_button('Filter', 'secondary', '', false); ?>
        <a href="<?php echo esc_url( admin_url('options-general.php?page=ctai-lite') ); ?>" class="button-link" style="margin-left:8px;">Reset to current month</a>
        <span class="description" style="margin-left:10px;">Showing items processed between <?php echo esc_html( wp_date('Y-m-d H:i T', $from_ts, ctai_lite_tz()) ); ?> and <?php echo esc_html( wp_date('Y-m-d H:i T', $to_ts, ctai_lite_tz()) ); ?>.</span>
    </form>

    <hr style="margin:28px 0;">
    <h2>Bulk Generate Alt Text (Missing Only)</h2>
    <p class="description">Process all images without alt text. You can start, pause, resume, or cancel. Progress updates live.</p>

    <div id="ctai-bulk-panel" class="card" style="max-width:900px;padding:16px;">
        <div style="display:flex;gap:8px;align-items:center;flex-wrap:wrap;margin-bottom:12px;">
            <button type="button" class="button button-primary" id="ctai-bulk-start">Start</button>
            <button type="button" class="button" id="ctai-bulk-pause">Pause</button>
            <button type="button" class="button" id="ctai-bulk-resume">Resume</button>
            <button type="button" class="button" id="ctai-bulk-cancel">Cancel</button>
            <button type="button" class="button" id="ctai-bulk-refresh">Refresh</button>
        </div>

        <div id="ctai-bulk-status" style="margin:8px 0 12px;"></div>

        <div style="background:#f0f0f1;border:1px solid #c3c4c7;height:16px;position:relative;border-radius:3px;overflow:hidden;">
            <div id="ctai-bulk-bar" style="background:#2271b1;height:100%;width:0%;transition:width .25s;"></div>
        </div>

        <div style="display:flex;gap:18px;margin-top:10px;">
            <div><strong>Total:</strong> <span id="ctai-bulk-total">0</span></div>
            <div><strong>Processed:</strong> <span id="ctai-bulk-processed">0</span></div>
            <div><strong>Skipped:</strong> <span id="ctai-bulk-skipped">0</span></div>
            <div><strong>Errors:</strong> <span id="ctai-bulk-errors">0</span></div>
        </div>

        <details style="margin-top:12px;">
            <summary>Show recent log</summary>
            <pre id="ctai-bulk-log" style="white-space:pre-wrap;background:#fff;border:1px solid #e2e2e2;padding:10px;max-height:220px;overflow:auto;"></pre>
        </details>
    </div>

    <?php
      // Items processed within the selected range
      $q = new WP_Query([
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'post_mime_type' => 'image',
        'posts_per_page' => 50,
        'fields'         => 'ids',
        'meta_key'       => CTAI_LITE_AMK_LAST_TS,
        'orderby'        => 'meta_value_num',
        'order'          => 'DESC',
        'meta_query'     => [
          [
            'key'     => CTAI_LITE_AMK_LAST_TS,
            'value'   => [$from_ts, $to_ts],
            'compare' => 'BETWEEN',
            'type'    => 'NUMERIC',
          ]
        ],
      ]);

      if ($q->have_posts()) {
        echo '<table class="widefat striped"><thead><tr>
          <th>Attachment</th><th>Status</th><th>Alt (len)</th>
          <th>Tokens In</th><th>Tokens Out</th><th>Cost</th><th>Sent Width</th><th>When</th><th>Last Error</th>
        </tr></thead><tbody>';
        foreach ($q->posts as $id) {
          $file = get_attached_file($id);
          $name = $file ? basename($file) : ('#'.$id);
          $alt  = (string) get_post_meta($id, CTAI_LITE_AMK_ALT, true);
          $len  = mb_strlen($alt);
          $in   = (int) get_post_meta($id, CTAI_LITE_AMK_TOK_IN, true);
          $out  = (int) get_post_meta($id, CTAI_LITE_AMK_TOK_OUT, true);
          $cost = (float) get_post_meta($id, CTAI_LITE_AMK_COST, true);
          $st   = (string) get_post_meta($id, CTAI_LITE_AMK_STATUS, true);
          $err  = esc_html((string) get_post_meta($id, CTAI_LITE_AMK_LAST_ERR, true));
          $ts   = (int) get_post_meta($id, CTAI_LITE_AMK_LAST_TS, true);
          $when = $ts ? wp_date('Y-m-d H:i:s T', $ts, ctai_lite_tz()) : '';
          $sent_w = (int) get_post_meta($id, CTAI_LITE_AMK_SENT_W, true);

          printf(
            '<tr><td><a href="%s">%s</a></td><td>%s</td><td>%s (%d)</td><td>%d</td><td>%d</td><td>$%0.5f</td><td>%s</td><td>%s</td><td>%s</td></tr>',
            esc_url(get_edit_post_link($id)),
            esc_html($name),
            esc_html($st ?: '-'),
            esc_html($alt ?: '-'),
            $len,
            $in,
            $out,
            $cost,
            $sent_w ? (int)$sent_w.'px' : '-',
            esc_html($when),
            $err ? $err : '&nbsp;'
          );
        }
        echo '</tbody></table>';
      } else {
        echo '<p>No items in this range.</p>';
      }
    ?>
</div>
<?php
}

/* ---------------- Connection test handler ---------------- */
add_action('admin_post_ctai_lite_test_connection', function () {
  if (!current_user_can('manage_options')) wp_die('Not allowed');
  if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'ctai_lite_test')) wp_die('Security check failed');

  $api_key = trim((string) get_option(CTAI_LITE_OPT_KEY, ''));
  if ($api_key === '') {
    wp_redirect(add_query_arg(['ctai_lite_msg'=>'no_key'], admin_url('options-general.php?page=ctai-lite')));
    exit;
  }

  $payload = [
    'model' => 'gpt-4o-mini',
    'messages' => [[ 'role' => 'user', 'content' => 'Reply with the word OK' ]],
    'max_tokens' => 2,
  ];

  $resp = wp_remote_post('https://api.openai.com/v1/chat/completions', [
    'headers' => [
      'Authorization' => 'Bearer ' . $api_key,
      'Content-Type'  => 'application/json',
    ],
    'body' => wp_json_encode($payload),
    'timeout' => 25,
  ]);

  if (is_wp_error($resp)) {
    wp_redirect(add_query_arg(['ctai_lite_msg'=>'err_http','detail'=>urlencode($resp->get_error_message())], admin_url('options-general.php?page=ctai-lite')));
    exit;
  }

  $body = json_decode(wp_remote_retrieve_body($resp), true);
  $ok = strtolower(trim((string)($body['choices'][0]['message']['content'] ?? '')));
  wp_redirect(add_query_arg(['ctai_lite_msg'=> ($ok==='ok' ? 'ok' : 'err_api')], admin_url('options-general.php?page=ctai-lite')));
  exit;
});

/* ---------------- Image: choose best rendition (URL + width) ---------------- */
function ctai_lite_best_image_source($attachment_id) {
  $maxw = (int) get_option(CTAI_LITE_OPT_MAX_WIDTH, ctai_lite_defaults()[CTAI_LITE_OPT_MAX_WIDTH]);
  if ($maxw < 320) $maxw = 320;

  $best = image_get_intermediate_size($attachment_id, array($maxw, $maxw));
  if (is_array($best) && !empty($best['url'])) {
    $w = isset($best['width']) ? (int)$best['width'] : null;
    return ['url' => $best['url'], 'width' => $w];
  }

  $candidate = null; $candW = null;
  foreach (['large', 'medium_large', 'medium', 'thumbnail'] as $size) {
    $src = wp_get_attachment_image_src($attachment_id, $size);
    if ($src && !empty($src[0])) {
      $w = isset($src[1]) ? (int)$src[1] : null;
      if ($w && $w <= $maxw + 50) return ['url'=>$src[0], 'width'=>$w];
      if (!$candidate) { $candidate = $src[0]; $candW = $w; }
    }
  }

  if ($candidate) return ['url'=>$candidate, 'width'=>$candW];

  $full = wp_get_attachment_url($attachment_id);
  return ['url'=>$full, 'width'=>null];
}

/* ---------------- Reusable server-side generator ---------------- */
function ctai_lite_generate_for_attachment($id){
  if (!$id || !wp_attachment_is_image($id)) return ['ok'=>false,'err'=>'Invalid attachment'];

  if ((int) get_option(CTAI_LITE_OPT_SKIP_IF_ALT, ctai_lite_defaults()[CTAI_LITE_OPT_SKIP_IF_ALT]) === 1) {
    $existing = (string) get_post_meta($id, CTAI_LITE_AMK_ALT, true);
    if ($existing !== '') {
      update_post_meta($id, CTAI_LITE_AMK_STATUS,  'skipped');
      update_post_meta($id, CTAI_LITE_AMK_LAST_ERR, '');
      update_post_meta($id, CTAI_LITE_AMK_TOK_IN,  0);
      update_post_meta($id, CTAI_LITE_AMK_TOK_OUT, 0);
      update_post_meta($id, CTAI_LITE_AMK_COST,    0);
      update_post_meta($id, CTAI_LITE_AMK_LAST_TS, time());
      return ['ok'=>true, 'alt'=>$existing];
    }
  }

  $key = trim((string) get_option(CTAI_LITE_OPT_KEY, ''));
  if ($key === '') return ['ok'=>false,'err'=>'Missing API key'];

  $asset   = ctai_lite_best_image_source($id);
  $img_url = $asset['url'];
  $sent_w  = $asset['width'];
  if (!$img_url) return ['ok'=>false,'err'=>'No image URL'];
  update_post_meta($id, CTAI_LITE_AMK_SENT_URL, esc_url_raw($img_url));
  if ($sent_w) update_post_meta($id, CTAI_LITE_AMK_SENT_W, (int)$sent_w);

  $chars  = (int) get_option(CTAI_LITE_OPT_CHARS, ctai_lite_defaults()[CTAI_LITE_OPT_CHARS]);

  $prompt = (string) get_option(CTAI_LITE_OPT_PROMPT, ctai_lite_defaults()[CTAI_LITE_OPT_PROMPT]);
  $prompt = str_replace('{MAX_CHARS}', (string)$chars, $prompt);

  if ((int) get_option(CTAI_LITE_OPT_USE_META, 0) === 1) {
    $meta_type = get_option(CTAI_LITE_OPT_META_TYPE, 'post_title');
    $post_id   = (int) get_post_field('post_parent', $id);
    $extra     = '';
    if ($post_id) {
      if ($meta_type === 'post_title') $extra = get_the_title($post_id);
      elseif ($meta_type === 'category') {
        $cats = get_the_category($post_id);
        $extra = !empty($cats) ? $cats[0]->name : '';
      }
    }
    if ($extra === '') $extra = get_bloginfo('name');
    $prompt .= ' Include context: ' . sanitize_text_field($extra) . '.';
  }

  $payload = [
    'model'    => 'gpt-4o-mini',
    'messages' => [[
      'role'    => 'user',
      'content' => [
        ['type' => 'text', 'text' => $prompt],
        ['type' => 'image_url', 'image_url' => ['url' => $img_url]],
      ],
    ]],
    'max_tokens' => 64,
  ];

  $resp = wp_remote_post('https://api.openai.com/v1/chat/completions', [
    'headers' => [
      'Authorization' => 'Bearer ' . $key,
      'Content-Type'  => 'application/json',
    ],
    'body'    => wp_json_encode($payload),
    'timeout' => 45,
  ]);

  if (is_wp_error($resp)) {
    update_post_meta($id, CTAI_LITE_AMK_STATUS, 'error');
    update_post_meta($id, CTAI_LITE_AMK_LAST_ERR, $resp->get_error_message());
    update_post_meta($id, CTAI_LITE_AMK_LAST_TS, time());
    return ['ok'=>false,'err'=>'HTTP error: '.$resp->get_error_message()];
  }

  $body = json_decode(wp_remote_retrieve_body($resp), true);
  if (!is_array($body)) {
    update_post_meta($id, CTAI_LITE_AMK_STATUS, 'error');
    update_post_meta($id, CTAI_LITE_AMK_LAST_ERR, 'Invalid JSON from API');
    update_post_meta($id, CTAI_LITE_AMK_LAST_TS, time());
    return ['ok'=>false,'err'=>'Invalid JSON from API'];
  }

  $alt = trim((string)($body['choices'][0]['message']['content'] ?? ''));
  if ($alt === '') {
    update_post_meta($id, CTAI_LITE_AMK_STATUS, 'error');
    update_post_meta($id, CTAI_LITE_AMK_LAST_ERR, 'Empty alt from API');
    update_post_meta($id, CTAI_LITE_AMK_LAST_TS, time());
    return ['ok'=>false,'err'=>'Empty alt from API'];
  }

  if (substr($alt, -1) === '.') $alt = mb_substr($alt, 0, mb_strlen($alt)-1);
  if (mb_strlen($alt) > $chars)  $alt = mb_substr($alt, 0, $chars);
  update_post_meta($id, CTAI_LITE_AMK_ALT, $alt);

  $in   = (int)($body['usage']['prompt_tokens']     ?? 0);
  $out  = (int)($body['usage']['completion_tokens'] ?? 0);
  $cost = $in * ctai_lite_rate_in() + $out * ctai_lite_rate_out();

  update_post_meta($id, CTAI_LITE_AMK_TOK_IN,  $in);
  update_post_meta($id, CTAI_LITE_AMK_TOK_OUT, $out);
  update_post_meta($id, CTAI_LITE_AMK_COST,    $cost);
  update_post_meta($id, CTAI_LITE_AMK_STATUS,  'success');
  update_post_meta($id, CTAI_LITE_AMK_LAST_ERR, '');
  update_post_meta($id, CTAI_LITE_AMK_LAST_TS, time());

  $ym     = wp_date('Y-m', null, ctai_lite_tz());
  $totals = (array) get_option(CTAI_LITE_OPT_MONTHLY, []);
  $totals[$ym] = isset($totals[$ym]) ? (float)$totals[$ym] + (float)$cost : (float)$cost;
  update_option(CTAI_LITE_OPT_MONTHLY, $totals, false);

  return ['ok'=>true, 'alt'=>$alt];
}

/* ---------------- UI hooks (Media modal + list view row) ---------------- */
add_filter('attachment_fields_to_edit', function ($form_fields, $post) {
  if (!wp_attachment_is_image($post->ID)) return $form_fields;
  $html  = '<button type="button" class="button button-primary ctai-lite-btn" data-attachment="'.(int)$post->ID.'">Generate Alt with AI</button>';
  $html .= ' <span class="ctai-lite-status" style="margin-left:8px;"></span>';
  $form_fields['ctai_lite'] = [
    'label' => __('Alt Text AI','ctai-lite'),
    'input' => 'html',
    'html'  => $html,
  ];
  return $form_fields;
}, 10, 2);

add_filter('media_row_actions', function ($actions, $post) {
  if (wp_attachment_is_image($post->ID)) {
    $actions['ctai_lite'] = '<a href="#" class="ctai-lite-row" data-attachment="'.(int)$post->ID.'">'.esc_html__('Generate Alt with AI','ctai-lite').'</a>';
  }
  return $actions;
}, 10, 2);

/* ---------------- JS & inline (Reveal + Bulk UI) ---------------- */
add_action('admin_enqueue_scripts', function ($hook) {
  // Load only on these admin pages
  $allowed = [
    'upload.php',
    'post.php',
    'media.php',
    'post-new.php',
    'settings_page_ctai-lite', // ✅ your settings page
  ];
  if (!in_array($hook, $allowed, true)) return;

  // Cache-bust version
  $js_path = plugin_dir_path(__FILE__) . 'ctai-lite.js';
  $js_ver  = file_exists($js_path) ? filemtime($js_path) : time();

  wp_enqueue_media();
  wp_enqueue_script(
    'ctai-lite',
    plugins_url('ctai-lite.js', __FILE__),
    ['jquery'],
    $js_ver,
    true
  );

  wp_localize_script('ctai-lite', 'CTAI_LITE', [
    'ajaxUrl' => admin_url('admin-ajax.php'),
    'nonce'   => wp_create_nonce('ctai_lite'),
    't'       => [
      'working' => __('Generating…','ctai-lite'),
      'ok'      => __('Alt updated.','ctai-lite'),
      'fail'    => __('Failed:','ctai-lite'),
    ],
  ]);

  // ✅ Only add inline JS on our settings page
  if ($hook === 'settings_page_ctai-lite') {

    // Optional Reveal button logic
    $inline = <<<JS
jQuery(function($){
  $(document).on('click','#ctai-reveal-key',function(e){
    e.preventDefault();
    var \$btn = $(this);
    var \$inp = $('input[name="ctai_lite_api_key"]');
    if (\$inp.attr('type') === 'text') {
      \$inp.attr('type','password').val('');
      \$btn.text('Reveal');
      return;
    }
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_reveal_key', nonce: CTAI_LITE.nonce }, function(res){
      if (res && res.success && res.data && res.data.key !== undefined) {
        \$inp.attr('type','text').val(res.data.key).focus();
        \$btn.text('Hide');
      }
    });
  });
});
JS;
    wp_add_inline_script('ctai-lite', $inline, 'after');

    // Bulk control + progress polling
    $bulk_js = <<<JS
jQuery(function($){
  var polling = null;

  function fmt(st){
    $('#ctai-bulk-status').text('Status: ' + (st.status || '-'));
    $('#ctai-bulk-total').text(st.total||0);
    $('#ctai-bulk-processed').text(st.processed||0);
    $('#ctai-bulk-skipped').text(st.skipped||0);
    $('#ctai-bulk-errors').text(st.errors||0);
    var pct = (st.total>0) ? Math.round((st.processed / st.total)*100) : 0;
    $('#ctai-bulk-bar').css('width', pct + '%');
    $('#ctai-bulk-log').text((st.log||[]).join("\\n"));
  }

  function getStatus(cb){
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_status', nonce: CTAI_LITE.nonce }, function(res){
      if (res && res.success) { fmt(res.data); if(cb) cb(res.data); }
    });
  }

  function tick(){
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_tick', nonce: CTAI_LITE.nonce }, function(res){
      if (res && res.success) {
        var st = res.data; fmt(st);
        if (st.status !== 'running') {
          if (polling) { clearInterval(polling); polling = null; }
        }
      } else {
        if (polling) { clearInterval(polling); polling = null; }
      }
    });
  }

  function ensurePolling(){
    if (!polling) {
      tick();
      polling = setInterval(tick, 1500);
    }
  }

  $('#ctai-bulk-start').on('click', function(e){
    e.preventDefault();
    if (polling) { clearInterval(polling); polling = null; }
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_start', nonce: CTAI_LITE.nonce }, function(res){
      if (res && res.success) { fmt(res.data); ensurePolling(); }
    });
  });

  $('#ctai-bulk-pause').on('click', function(e){
    e.preventDefault();
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_pause', nonce: CTAI_LITE.nonce }, function(){ getStatus(); });
  });

  $('#ctai-bulk-resume').on('click', function(e){
    e.preventDefault();
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_resume', nonce: CTAI_LITE.nonce }, function(res){
      if (res && res.success) { fmt(res.data); ensurePolling(); }
    });
  });

  $('#ctai-bulk-cancel').on('click', function(e){
    e.preventDefault();
    if (!confirm('Cancel and clear the current queue?')) return;
    if (polling) { clearInterval(polling); polling = null; }
    $.post(CTAI_LITE.ajaxUrl, { action:'ctai_lite_bulk_cancel', nonce: CTAI_LITE.nonce }, function(){ getStatus(); });
  });

  $('#ctai-bulk-refresh').on('click', function(e){
    e.preventDefault();
    getStatus();
  });

  getStatus(function(st){ if (st.status==='running') ensurePolling(); });
});
JS;
    wp_add_inline_script('ctai-lite', $bulk_js, 'after');
  }
});

/* ---------------- AJAX: generate (uses helper) ---------------- */
add_action('wp_ajax_ctai_lite_generate', function () {
  try {
    if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
    check_ajax_referer('ctai_lite', 'nonce');
    $id = isset($_POST['attachment_id']) ? (int) $_POST['attachment_id'] : 0;
    $res = ctai_lite_generate_for_attachment($id);
    if ($res['ok']) wp_send_json_success(['alt' => $res['alt']]);
    wp_send_json_error(['message' => $res['err'] ?: 'Unknown error'], 500);
  } catch (Throwable $e) {
    wp_send_json_error(['message'=>'Server error: '.$e->getMessage()], 500);
  }
});

/* ---------------- AJAX: reveal key (admin only) ---------------- */
add_action('wp_ajax_ctai_lite_reveal_key', function () {
  if (!current_user_can('manage_options')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');
  $key = (string) get_option(CTAI_LITE_OPT_KEY, '');
  wp_send_json_success(['key' => $key]);
});

/* ---------------- Auto-generate on upload ---------------- */
add_filter('wp_generate_attachment_metadata', function ($metadata, $attachment_id) {
  if (!wp_attachment_is_image($attachment_id)) return $metadata;
  if (trim((string) get_option(CTAI_LITE_OPT_KEY, '')) === '') return $metadata;
  ctai_lite_generate_for_attachment($attachment_id);
  return $metadata;
}, 10, 2);

/* ---------------- AJAX: bulk start ---------------- */
add_action('wp_ajax_ctai_lite_bulk_start', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');

  // Build a queue of attachments missing alt text (limit generously)
  $ids = [];
  $q = new WP_Query([
    'post_type'      => 'attachment',
    'post_status'    => 'inherit',
    'post_mime_type' => 'image',
    'fields'         => 'ids',
    'posts_per_page' => 2000, // adjust if needed
    'meta_query'     => [
      'relation' => 'OR',
      [
        'key'     => '_wp_attachment_image_alt',
        'compare' => 'NOT EXISTS'
      ],
      [
        'key'     => '_wp_attachment_image_alt',
        'value'   => '',
        'compare' => '='
      ],
    ],
  ]);
  if ($q->have_posts()) $ids = $q->posts;

  $st = [
    'status'     => 'running',
    'queue'      => array_values($ids),
    'total'      => count($ids),
    'processed'  => 0,
    'skipped'    => 0,
    'errors'     => 0,
    'current_id' => null,
    'started_at' => time(),
    'last_ts'    => time(),
    'log'        => [],
  ];
  if (empty($ids)) {
    $st['status'] = 'done';
    $st['log'][]  = wp_date('H:i:s') . ' — No images found that are missing alt.';
  } else {
    $st['log'][]  = wp_date('H:i:s') . ' — Queue created with ' . count($ids) . ' items.';
  }
  ctai_lite_bulk_set_state($st);
  wp_send_json_success($st);
});

/* ---------------- AJAX: bulk tick (process a small batch) ---------------- */
add_action('wp_ajax_ctai_lite_bulk_tick', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');

  $st = ctai_lite_bulk_get_state();
  if ($st['status'] !== 'running') {
    $st['last_ts'] = time();
    ctai_lite_bulk_set_state($st);
    wp_send_json_success($st);
  }

  $batch = 5; // items per tick
  for ($i=0; $i<$batch; $i++) {
    if ($st['status'] !== 'running') break;
    $next = array_shift($st['queue']);
    if (!$next) { // queue drained
      $st['status'] = 'done';
      $st['current_id'] = null;
      $st['log'][] = wp_date('H:i:s') . ' — Completed. Total processed: ' . $st['processed'] . '.';
      break;
    }
    $st['current_id'] = (int)$next;

    // Respect "skip if alt" (generator already does)
    $res = ctai_lite_generate_for_attachment($st['current_id']);
    if ($res['ok']) {
      $has = get_post_meta($st['current_id'], '_wp_attachment_image_alt', true);
      if ($has === '') {
        $st['skipped']++;
        $st['log'][] = 'Skipped #' . $st['current_id'];
      } else {
        $st['processed']++;
        $st['log'][] = 'OK #' . $st['current_id'] . ' — ' . esc_html( mb_strimwidth($has, 0, 60, '…') );
      }
    } else {
      $st['errors']++;
      $st['log'][] = 'Error #' . $st['current_id'] . ': ' . $res['err'];
    }
  }

  $st['current_id'] = null;
  $st['last_ts'] = time();
  ctai_lite_bulk_set_state($st);
  wp_send_json_success($st);
});

/* ---------------- AJAX: bulk pause ---------------- */
add_action('wp_ajax_ctai_lite_bulk_pause', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');
  $st = ctai_lite_bulk_get_state();
  if ($st['status'] === 'running') {
    $st['status'] = 'paused';
    $st['log'][]  = wp_date('H:i:s') . ' — Paused.';
    ctai_lite_bulk_set_state($st);
  }
  wp_send_json_success($st);
});

/* ---------------- AJAX: bulk resume ---------------- */
add_action('wp_ajax_ctai_lite_bulk_resume', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');
  $st = ctai_lite_bulk_get_state();
  if (in_array($st['status'], ['paused','idle'], true)) {
    if (!empty($st['queue'])) {
      $st['status'] = 'running';
      $st['log'][]  = wp_date('H:i:s') . ' — Resumed.';
    } else {
      $st['status'] = 'done';
      $st['log'][]  = wp_date('H:i:s') . ' — Nothing to resume.';
    }
    ctai_lite_bulk_set_state($st);
  }
  wp_send_json_success($st);
});

/* ---------------- AJAX: bulk cancel ---------------- */
add_action('wp_ajax_ctai_lite_bulk_cancel', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');
  $st = ctai_lite_bulk_get_state();
  $st['status']    = 'idle';
  $st['queue']     = [];
  $st['current_id']= null;
  $st['log'][]     = wp_date('H:i:s') . ' — Cancelled and cleared queue.';
  ctai_lite_bulk_set_state($st);
  wp_send_json_success($st);
});

/* ---------------- AJAX: bulk status ---------------- */
add_action('wp_ajax_ctai_lite_bulk_status', function () {
  if (!current_user_can('upload_files')) wp_send_json_error(['message'=>'Not allowed'], 403);
  check_ajax_referer('ctai_lite', 'nonce');
  wp_send_json_success( ctai_lite_bulk_get_state() );
});
